#ifndef CE2_ECPKI_TYPES_H
#define CE2_ECPKI_TYPES_H
/*
   *  Object % CE2_ECPKI_Types.h    : %
   *  State           :  %state%
   *  Creation date   :  05/12/2005
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief Contains all of the enums and definitions 
   *         that are used for the CE2 ECPKI APIs, as well as the APIs themselves.
   *
   *  \version CE2_ECPKI_Types.hh#1:incl:1
   *  \author R.Levin
   *  \remarks Copyright (C) 2006 by Discretix Technologies Ltd.
   *           All rights reserved.
   */

/*
 * All the includes that are needed for code using this module to
 * compile correctly should be #included here.
 */
//#include "CE2_public.h"



#ifdef __cplusplus
extern "C"
{
#endif


 
/************************ Defines ******************************/

/*! The valid maximum EC modulus size in 32-bit words */
#define CE2_ECPKI_MODUL_MAX_LENGTH_IN_WORDS  18 /*(CE2_ECPKI_MODUL_MAX_LENGTH_IN_BITS + 31)/(sizeof(DxUint32_t))*/

/* The type defines integer array of lengths of maximum lengths of EC modulus */ 
typedef  DxUint32_t  CE2_ECPKI_ARRAY_t[CE2_ECPKI_MODUL_MAX_LENGTH_IN_WORDS];

/*! Predefined arithmetic reduction mode for modular multiplication:
   Choose Montgomery or Barret. Note: Barret is not currently supported. */
#define CE2_ECPKI_DEFINED_REDUCTION_MODE   Montgomery  /* Barret */

#define ECPKI_USER_KEY_MAX_BUFFER_SIZE 256

/**************************************************************************************
 *	              Enumerators
 ***************************************************************************************/

/* Enumerator for the EC finite field Identificator */
typedef enum
{
	CE2_GFp = 0,
	CE2_GF2 = 1,
	
	CE2_FieldID_OffMode
			
} CE2_ECPKI_FieldID_t;

/*------------------------------------------------------------------*/
/* Enumerator for the EC Domain Identificator 
   References: [13] - SEC 2: Recomended elliptic curve domain parameters. 
                      Version 1.0. Certicom 2000. 
               [8]  - WAP-261-WTLS-20010406-a, Version 06-April-2001     */

typedef enum
{
   /* For prime field */
   CE2_ECPKI_DomainID_EMPTY	    = 0,
   CE2_ECPKI_DomainID_WMDRM10	    = 1,   /* EC from WM DRM 10.05  (prime 160 bit field) */

   CE2_ECPKI_DomainID_secp160k1    = 2,   /* EC secp160r1 from [13] (identic 7 from [8]), used in CE2 3.5 */
   CE2_ECPKI_DomainID_secp160r1    = 3,   /* EC secp160k1 from [13]  */
   CE2_ECPKI_DomainID_secp160r2    = 4,   /* EC secp160r2 from [13]  */
   CE2_ECPKI_DomainID_secp192k1    = 5,   /* EC secp192k1 from [13]  */
   CE2_ECPKI_DomainID_secp192r1    = 6,   /* EC secp192r1 from [13]  */
   CE2_ECPKI_DomainID_secp224k1    = 7,   /* EC secp224k1 from [13]  */
   CE2_ECPKI_DomainID_secp224r1    = 8,   /* EC secp224r1 from [13]  */
   CE2_ECPKI_DomainID_secp256k1    = 9,   /* EC secp256k1 from [13]  */
   CE2_ECPKI_DomainID_secp256r1    = 10,  /* EC secp256r1 from [13]  */
   CE2_ECPKI_DomainID_secp384r1    = 11,  /* EC secp384r1 from [13]  */
   CE2_ECPKI_DomainID_secp521r1    = 12,  /* EC secp521r1 from [13]  */
   
   /* For base 2 field - now not implemented*/


   /* EC domain recieved from user by pointer */
   CE2_ECPKI_DomainID_NotStandard,        /* EC domain recieved from user by pointer */
      
   CE2_ECPKI_DomainID_OffMode   = 13        /* Set 13 because now not all of data 
										      is filled in LLF_Domains */    
}CE2_ECPKI_DomainID_t;   


/*------------------------------------------------------------------*/
/* Defines the enum for the HASH operation mode.
 * The enumerator defines 6 HASH modes according to IEEE 1363.
 * In the WMDRM DSA is used SHA1 mode only.
 */
typedef enum
{   
   CE2_ECPKI_HASH_SHA1_mode    = 0,
   CE2_ECPKI_HASH_SHA224_mode  = 1,
   CE2_ECPKI_HASH_SHA256_mode  = 2,
   CE2_ECPKI_HASH_SHA384_mode  = 3,
   CE2_ECPKI_HASH_SHA512_mode  = 4,

   CE2_ECPKI_After_HASH_SHA1_mode    = 5, /* When the data is already hashed with SHA1 */
   CE2_ECPKI_After_HASH_SHA224_mode  = 6, /* When the data is already hashed with SHA224 */
   CE2_ECPKI_After_HASH_SHA256_mode  = 7, /* When the data is already hashed with SHA256 */
   CE2_ECPKI_After_HASH_SHA384_mode  = 8, /* When the data is already hashed with SHA384 */
   CE2_ECPKI_After_HASH_SHA512_mode  = 9, /* When the data is already hashed with SHA512 */

   CE2_ECPKI_HASH_NumOfModes

}CE2_ECPKI_HASH_OpMode_t;   
  

/*---------------------------------------------------*/
/* Enumerator for the EC point coordinates identificator */
typedef enum
{   
   CE2_EC_PointAffine           = 0,   /* Affine coordinates */
   CE2_EC_PointJacobian         = 1,   /* Jacobian projective coordinates */
   CE2_EC_PointProjStandard     = 2,   /* Standard projective coordinates */
   CE2_EC_PointChudnovsky       = 3,   /* Chudnovsky projective coordinates */
   
   CE2_EC_PointID_OffMode       = 4,
   
}CE2_ECPKI_PointID_t; 

/*---------------------------------------------------*/
/* Enumerator for the EC point compression identificator */
typedef enum
{   
   CE2_EC_PointCompressed     = 2,   
   CE2_EC_PointUncompressed   = 4,      
   CE2_EC_PointContWrong      = 5, /* wrong Point Control value */ 
   CE2_EC_PointHybrid         = 6,

   CE2_EC_PointCompresOffMode = 8   
}CE2_ECPKI_PointCompression_t; 

/*----------------------------------------------------*/
/*  Enumerator for compatibility ofthe DHC 
   with cofactor multiplication to DH ordinary */
typedef enum
{
   CE2_ECPKI_SVDP_DHC_CompatibleDH	        = 0,   /* Requested compatiblity of SVDP_DHC with cofactor to SVDP_DH */
   CE2_ECPKI_SVDP_DHC_NonCompatibleDH	    = 1,   /* Compatiblity of SVDP_DHC with cofactor to SVDP_DH is not requested */
   
   CE2_ECPKI_SVDP_DHC_OffMode 
    
}CE2_ECPKI_SVDP_DHC_OpMode_t; 

/*----------------------------------------------------*/
/* Enumerator for choosing reduction mode on EC arithmetic operations */
typedef enum {
   CE2_Montgomery   = 0,   /*  */
   CE2_Barret       = 1,   /*  */
   
   CE2_ReductionOffMode 
    
}CE2_ReductionMode_t; 

#define DEFINED_REDUCTION_MODE  Montgomery

/*----------------------------------------------------*/
/* Enumerator for indication of EC adding or subtracting operation on FullAdd algotithm */
typedef enum {
   CE2_Add            = 0,   /*  */
   CE2_Subtract       = 1,   /*  */
      
   CE2_AddSubtract_OffMode 
    
}CE2_AddSubtractMode_t; 


/***************************************************************************************
 *	             EC arithmetic operations mode                                    
 ***************************************************************************************/
typedef struct {
   CE2_ECPKI_FieldID_t      Field;	  
   CE2_ReductionMode_t           ReductionMode;

} CE2_ECPKI_ArithmOpMode_t;

/**************************************************************************************
 *	             EC DomainInfo structure definition
 ***************************************************************************************/
/*! This structure contains the EC domain data used on the CE2 level. */
typedef  struct {		/*! Size of field modulus  in bits and in words*/
	DxUint16_t      EC_ModulusSizeInBits;
	/*! Sizes of order of generator in bits and in words*/
	DxUint16_t      EC_OrderSizeInBits;
	/*! EC Domain identifier: number of curve*/
	CE2_ECPKI_DomainID_t	DomainID;
	/*! EC finite field identifier*/
	CE2_ECPKI_FieldID_t	FieldID;
} CE2_ECPKI_DomainInfo_t ;    


/**************************************************************************************
 *	             EC  point structures definitions
 ***************************************************************************************/
					 
/*! The structure containing the EC point in affine coordinates 
   and little endian form */
typedef  struct  
{
    /* Point coordinates */	  	
    /*! X-coordinate */
    CE2_ECPKI_ARRAY_t			CoordX;     
    /*! Y-coordinate */
    CE2_ECPKI_ARRAY_t			CoordY;     
         
}CE2_ECPKI_PointAffine_t;


/*! The structure containing the EC point in any type of coordinates,
   defined by PointID. */
typedef  struct CE2_ECPKI_Point_t 
{
 	  	
    /*! X-coordinate */
    CE2_ECPKI_ARRAY_t			CoordX;
    /*! Y-coordinate */
    CE2_ECPKI_ARRAY_t			CoordY;
    /*! Z-coordinate */
    CE2_ECPKI_ARRAY_t			CoordZ;     
    /*! Enum variable: defines type of point presentation*/
    CE2_ECPKI_PointID_t		PointID;   
}CE2_ECPKI_Point_t;



/**************************************************************************************
 *	              ECPKI public and private key  Structures
 ***************************************************************************************/



/*! The user structure containing EC public key data base form.
 
   Size = 2*CE2_ECPKI_MODUL_MAX_LENGTH_IN_WORDS + 2 + 
   ( sizeof(LLF_ECPKI_publ_key_db_def.h) = 0 )          */

typedef struct   CE2_ECPKI_UserPublKey_t 
{   
   DxUint32_t 			valid_tag;  
   DxUint8_t  			PublKeyDbBuff[ECPKI_USER_KEY_MAX_BUFFER_SIZE];
   CE2_ECPKI_DomainID_t DomainID;
   
}  CE2_ECPKI_UserPublKey_t;


/*! The user structure containing EC private key data base in little endian form.
 
   Size = CE2_ECPKI_MODUL_MAX_LENGTH_IN_WORDS + 3 + 
   ( sizeof(LLF_ECPKI_priv_key_db_def.h) = 0 )          */
typedef struct   CE2_ECPKI_UserPrivKey_t 
{   
   DxUint32_t 					valid_tag;  
   DxUint8_t  					PrivKeyDbBuff[ECPKI_USER_KEY_MAX_BUFFER_SIZE];
   CE2_ECPKI_DomainID_t DomainID;
    
}  CE2_ECPKI_UserPrivKey_t;

/*****************************************************************************/ 

  
  

#ifdef __cplusplus
}
#endif


#endif
